/*jshint esversion: 6 */
/* global define */
define([	"lib/Zoot",		"src/build/Layer", 	"src/math/Random",	"lodash", "lib/tasks"],
function(	Z,				Layer, 				Random,				lodash,   tasks) {
	"use strict";

/**
 *
**/

	const kPrefix = "Adobe.Box2d",
		  kStart_Immediately = 0,
		  kStart_WaitForTrigger = 1,
          kShape_Contour = 0,
          kShape_Fast = 1,
		  kNone = 0, // TODO: share these with the Box2dWorld behavior
		  kCreate = 1,
		  kDestroy = 2;

	function getSceneLayer (args) {
		var tracks = args.stageLayer.privateLayer.getScene().getChildren(),
			layer = tracks[0].getChildFromIndex(0).getSdkLayer();
		return layer;
	}

	function storeDangleLayer (args, layer) {
		let sceneLayer = getSceneLayer(args);

		let tBox2d = sceneLayer.getSharedSceneData(kPrefix);

		if (tBox2d) {
			tBox2d.dangleLayers.set(layer, layer);
		} else {
			tBox2d = {
				dangleLayers: new Map(),
				bodies: new Map()
			};
			tBox2d.dangleLayers.set(layer, layer);
		}

		sceneLayer.setSharedSceneData(kPrefix, tBox2d);
	}

	function addBody (layer, body, overwrite) {

		let tBox2d = layer.getSharedSceneData(kPrefix);

		if (tBox2d) {
			if (overwrite || !tBox2d.bodies.has(body.layer)) {
				tBox2d.bodies.set(body.layer, body);
			}
		} else {
			tBox2d = {
				dangleLayers : new Map(),
				bodies : new Map()
			};

			tBox2d.bodies.set(body.layer, body);
		}

		layer.setSharedSceneData(kPrefix, tBox2d);
	}

	function validRigidBody (layer) {
		if (layer.privateLayer.getWarpWithParent()) {
			// Warp dependent puppets are just storing params, so return true
			return true;
		}

		// Check if group has layers
		// Currently doing this by getting the outline and ensuring it has area
		// FIXME: There has to be a cleaner way to do this
		let meshPoints = layer.privateLayer.getWarperContainer().getGeometryOutline();

		let minp = 0,
		maxp = 0;

		if (meshPoints.length > 0) {
			minp = meshPoints[0][0];
			maxp = meshPoints[0][0];

			meshPoints.forEach(function (p) {
				if (p[0] < minp) minp = p[0];
				if (p[0] > maxp) maxp = p[0];
			});
		}

		if (minp === maxp) {
			return false;
		}

		return true;
	}

	function setupBodyParams (layer, mData, args, self) {

		mData.dynamic = mData.hasOwnProperty("dynamic") ? mData.dynamic : false;
		mData.collide = mData.collide || false;

		// Layers with dangle handles can't be dynamic
		let tBox2d = getSceneLayer(args).getSharedSceneData(kPrefix);
		if (tBox2d && tBox2d.dangleLayers.has(layer)) {
			mData.dynamic = false;
		}

		// Dependent puppets, or the root puppet, are only param holders
		// Also, if it is not dynamic or collide (can happen when a dynamic only layer also has dangle handles), then make it param only.
		let paramOnly = layer.privateLayer.getWarpWithParent() || !layer.privateLayer.parentPuppet || (!mData.dynamic && !mData.collide);

		let gravDir = -args.getParam("gravityDirection") * Math.PI / 180.0,
			mag = args.getParam("gravityStrength");

		let nextAction = kNone;
		self.bFirstTrigger = false;
		if (args.getParam("startWhen") === kStart_Immediately) {
			nextAction = kCreate;
			self.bFirstTrigger = true;
		}

		let bodyDef = {
			body : false,
			layer : layer,
			paramOnly : paramOnly,
			trackItem : paramOnly ? false : layer.privateLayer.getTrackItem(),
			density : args.getParam("density") / 100.0,
			dynamic : mData.dynamic,
			collidable : mData.collide,
			autoAddChildren : false,
			// FIXME: currently looking at the privateWarper object to see how many anchors (handles) there are.
			// Deformable bodies need more than one. Should this be a cpp callback, or just store it in the regular warperContainer object instead of the private one?
			deformable : !mData.dynamic && mData.collide,
			motorStrength : args.getParam("motorStrength") / 10.0,
			motorBody : undefined,
			//source : false,
			approxShape : args.getParam("GeometryType") === kShape_Fast,
			friction : args.getParam("friction") / 100.0,
			restitution : args.getParam("restitution") / 100.0,
			wallCollide : args.getParam("wallCollide"),
			beingAltered : new Map(),
			attachedToBehavior : true,
			action : nextAction,
			controlJoints : new Map(),
			controlJointBody : undefined,
			beingDragged : new Map(),
			behaviorLayer : args.stageLayer,
			gravVec: [-Math.sin(gravDir) * mag, Math.cos(gravDir) * mag]
		};

		return bodyDef;
	}

	var kAbout = "$$$/private/animal/Behavior/RigDynamics/About=Physics, (c) 2014.";
		//mat3 = Z.Mat3;
		//arrayPush = Array.prototype.push,
		//kNoAuto = { translation : false, linear : false };

	function defineDangleParams() {
		return [
				{
					id: "targetHandles", type: "handle", uiName: "$$$/animal/Behavior/RigDynamics/Parameter/targetHandles=Dangle Handles",
					dephault: { match: "//Adobe.RigDynamics.Dangle" }
				},
				{
					id: "stiffness", type: "slider", uiName: "$$$/animal/Behavior/RigDynamics/Parameter/stiffness=Spring Stiffness", uiUnits: "%",
					min:1, max:200, precision:0, dephault:100
				},
				{
					id: "windStrength", type: "slider", uiName: "$$$/animal/Behavior/RigDynamics/Parameter/windStrength=Wind Strength", uiUnits: "%", min: 0, precision: 1, dephault: 0
				},
				{
					id: "windDirection", type: "angle", uiName: "$$$/animal/Behavior/RigDynamics/Parameter/windDirection=Wind Direction", precision: 0, dephault: 90
				},
				{
					id: "windVariation", type: "slider", uiName: "$$$/animal/Behavior/RigDynamics/Parameter/windVariation=Wind Variation", uiUnits: "%", min: 0, max: 100, precision: 1, dephault: 0
				}
		];
	}

	function defineCollisionParams () {
		return [
			{
                id: "startWhen", type: "enum", uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/startWhen=Start", dephault: kStart_Immediately,
				items: [
							{ id: kStart_WaitForTrigger, 		uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/startWhen/WhenTriggered=When Triggered" },
							{ id: kStart_Immediately, 			uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/startWhen/startImmediately=Immediately" },
					   ],
				uiToolTip: "$$$/animal/behavior/Box2d/Body/Parameter/startwhen/tooltip=Choose between starting immediately or waiting for a specific triggering key press assigned to this puppet or one of its parents"
			},
			{
				id: "GeometryType", type: "enum", uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/GeometryType=Shape", dephault: kShape_Contour,
				items: [
                            { id: kShape_Fast,      uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/GeometryType/Fast=Fast" },
                            { id: kShape_Contour,   uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/GeometryType/Contour=Contour" }
                       ],
					uiToolTip: "$$$/animal/Behavior/Box2d/Body/Parameter/GeometryType/tooltip=Adjust how precise collisions should be (Fast has better performance, Contour has better edge detection)", hideRecordButton: true
			},
			{
				id: "density", type: "slider", uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/weight=Weight",
				min:1, max:1000, precision:1, dephault:100,
				uiToolTip: "$$$/animal/Behavior/Box2d/Body/Parameter/density/tooltip=Density of layers that collide with each other. Higher numbers are heavier"
			},
			{
				id: "friction", type: "slider", uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/friction=Friction",
				min:0, max:100, precision:1, dephault:30, uiUnits: "%",
				uiToolTip: "$$$/animal/Behavior/Box2d/Body/Parameter/friction/tooltip=Adjust how much colliding layers decelerate when sliding against each other"
			},
			{
				id: "restitution", type: "slider", uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/restitution=Bounciness",
				min:0, max:100, precision:1, dephault:20, uiUnits: "%",
				uiToolTip: "$$$/animal/Behavior/Box2d/Body/Parameter/restitution/tooltip=Adjust how much colliding layers bounce"
			},
			{
				id: "wallCollide", type: "checkbox", uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/wallCollide=Bounce Off Scene Sides", dephault:true,
				uiToolTip: "$$$/animal/Behavior/Box2d/Body/Parameter/wallCollide/tooltip=Make colliding layers bounce when hitting scene bounds"
			},
			{
				id: "motorStrength", type: "slider", uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/motorStrength=Return Strength",
				min:0, max:100, precision:1, dephault:0, uiUnits: "%",
				uiToolTip: "$$$/animal/Behavior/Box2d/Body/Parameter/motorStrength/tooltip=Adjust how quickly dynamic layers return to their original position (0 = do not return)"
			},
			{
				id: "collideLayers", type: "layer", uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/collideLayers=Collide Layers", dephault: { match: "//Adobe.Physics.Collide"}
			},
			{
				id: "dynamicLayers", type: "layer", uiName: "$$$/animal/Behavior/Box2d/Body/Parameter/dynamicLayers=Dynamic Layers", dephault: { match: "//Adobe.Physics.Dynamic"}
			},
		];
	}

	function onCreateRigDynamicsStageBehavior (self, args) {
		self.rand = new Random(123456789);

		self.draggableHandles = args.getStaticParam("draggableHandles");
		lodash.forEach(self.draggableHandles, function (h) {
			h.isDraggable = true;
		});

		self.fixedHandles = args.getStaticParam("fixedHandles");
		lodash.forEach(self.fixedHandles, function (h) {
			h.isFixed = true;
		});

		self.dangleHandles = args.getStaticParam("targetHandles");
		lodash.forEach(self.dangleHandles, function (h) {
			h.isDangle = true;
			h.springStiffness = Math.pow(1.05,100);
			h.dangleGravity = [0,10];
			h.dangleWind = [0,0];
			// console.logToUser("dangle handle: " + h.getName());
		});
	}

	function onCreateCollisionStageBehavior (self, args) {
		let collideLayers = args.getStaticParam("collideLayers");
		let dynamicLayers = args.getStaticParam("dynamicLayers");

		self.taggedLayers = collideLayers.concat(dynamicLayers);

		// Check our own layer, because it is not matched
		let tags = args.stageLayer.privateLayer.getLayerTags(),
			tagDynamic = !lodash.isUndefined(tags["Adobe.Physics.Dynamic"]),
			tagCollide = !lodash.isUndefined(tags["Adobe.Physics.Collide"]);

		if ((tagDynamic || tagCollide)) {
			self.taggedLayers.push(args.stageLayer);
		}

		self.shouldInit = true;
	}

	function onRigDynamicsAnimate (self, args) {
		// per handle physics properties
		var springStiffness = Math.pow(1.05,args.getParam("stiffness"));
		// per sim layer physics properties; TODO: living on handles for now as well
		var windAngle = 2*Math.PI/360*args.getParam("windDirection") - Math.PI/2 + args.getParam("windVariation")/100*(1-2*self.rand.random())*Math.PI/2,
			windMag = args.getParam("windStrength"),
			gravAngle = 2*Math.PI/360*args.getParam("gravityDirection") - Math.PI/2,
			gravMag = args.getParam("gravityStrength"),
			dangleGravity = [gravMag*Math.cos(gravAngle),gravMag*Math.sin(gravAngle)],
			dangleWind = [windMag*Math.cos(windAngle),windMag*Math.sin(windAngle)];

		lodash.forEach(self.dangleHandles, function (h) {
			h.springStiffness = springStiffness;
			h.dangleGravity = dangleGravity;
			h.dangleWind = dangleWind;
			// console.logToUser("dangle handle: " + h.getName());
		});
	}

	function onCollisionAnimate (self, args) {
		let sceneLayer = getSceneLayer(args);

		if (self.shouldInit) {
			// we are adding all puppets with Box2dBody behavior to Box2d scene object
			// follow this approach instead of RigDynamics/World approach that pollutes handle objects.

			lodash.forEach(self.dangleHandles, function (h) {
				// Store the warper layer of each dangle handle, so that
				// we can filter out any dynamic layer tags applied to it.
				storeDangleLayer(args, h.getWarperLayer().getSdkLayer());
			});

			// Add the tagged layers
			self.taggedLayers.forEach( function (layer) {

				let tags = layer.privateLayer.getLayerTags(),
					tagDynamic = !lodash.isUndefined(tags["Adobe.Physics.Dynamic"]),
					tagCollide = !lodash.isUndefined(tags["Adobe.Physics.Collide"]);

				if ((tagDynamic || tagCollide) && validRigidBody(layer)) {
					let mData = {dynamic: tagDynamic, collide: tagCollide};

					let bodyDef = setupBodyParams(layer, mData, args, self);

					// Overwrite only if this is our layer (behavior was applied down the tree)
					addBody(sceneLayer, bodyDef, layer === args.stageLayer);
				}
			});

			self.shouldInit = false;
		} else {
			// Update params
			let tBox2d = sceneLayer.getSharedSceneData(kPrefix);

			if (!tBox2d) return;

			let triggerType = args.getTriggeringType(),
				bActualTrigger = (triggerType === "realTrigger" || triggerType === "behaviorTrigger");       // not a re-trigger at our own request

			self.taggedLayers.forEach( function (layer) {
				let bodyDef = tBox2d.bodies.get(layer);

				if (bodyDef) {
					bodyDef.density = args.getParam("density") / 100.0;
					bodyDef.friction = args.getParam("friction") / 100.0;
					bodyDef.restitution = args.getParam("restitution") / 100.0;
					bodyDef.motorStrength = args.getParam("motorStrength") / 10.0;
					bodyDef.wallCollide = args.getParam("wallCollide");

					let gravDir = -args.getParam("gravityDirection") * Math.PI / 180.0,
						mag = args.getParam("gravityStrength");

					bodyDef.gravVec = [-Math.sin(gravDir) * mag, Math.cos(gravDir) * mag];

					if (self.bTrigger && !self.bFirstTrigger) {
						if (!bActualTrigger) {
							bodyDef.action = kDestroy;
						}
					} else if (bActualTrigger) {
						if (!self.bFirstTrigger) {
							bodyDef.action = kCreate;
						}
					}
				}
			});

			if (self.bTrigger && !self.bFirstTrigger) {
				if (!bActualTrigger) {
					self.bTrigger = false;
				}
			} else if (bActualTrigger) {
				self.bTrigger = true;

				self.bFirstTrigger = false;
			}

			tBox2d.bodies.forEach(function (bi) {
				if (bi.deformable || bi.paramOnly) return;

				// Only check dragging for layers that this behavior owns, so that triggers
				// work correctly
				if (bi.behaviorLayer !== args.stageLayer) return;

				let rootHandle = bi.layer.privateLayer.getHandleTreeRoot();

				rootHandle.breadthFirstEach(function (handle) {
					let hasTasks = tasks.handle.internal.hasPendingTasks(handle),
						frameChanged = bi.prevMDof ? !tasks.handle.internal.sameMatrixDof(handle, bi.prevMDof.get(handle)) : false;

					//console.logToUser(`{bi.layer.privateLayer.name}, ${hasTasks}`);

					if (frameChanged || hasTasks) {
						bi.beingDragged.set(handle, true);
					} else {
						// Not being dragged, but add a no-op task here so that the warper doesn't return it to rest. If we don't then
						// Box2dWorld can't get the correct world handle transform.
						bi.beingDragged.set(handle, false);

						// Do not set when the layer is being destroyed (untriggered). Then we actually want to reset it.
						// Instead of set/get Frame, use task with 0 weight. That leaves it for the task mediation step
						// to preserve current value, and won't cause jittering.
						let pLayer = bi.layer.privateLayer,
							isFree = pLayer.motionMode.translation === true && pLayer.motionMode.linear === true;

						// FIXME: This logic should be simplified. The second clause seems to be necessary, or else
						// the handle positions that are read in Box2dWorld will be incorrect when they are on a
						// free falling rigid body and Box2dWorld is trying to read the world position of a draggable
						// handle that is not the root handle. If we don't touch root handles in chain puppets attached
						// as hinge or weld, there are problems there.
						if (bi.action !== kDestroy && !(handle === rootHandle && isFree)) {
							let tdof = {x: 0, y: 0};
							let move = new tasks.MoveTo(tdof);
							tasks.handle.attachTask(handle, move, 0.0);
						}
					}
				});
			});

			sceneLayer.setSharedSceneData(kPrefix, tBox2d);
		}
	}

	return {
		about: kAbout,
		description: "$$$/animal/Behavior/RigDynamics/Desc=Enable physics simulations to move handles naturally in reponse to movement and wind, collide and bounce layers off each other, and both to be affected by gravity",
		uiName: 	"$$$/animal/Behavior/RigDynamics/UIName=Physics",
		defaultArmedForRecordOn: true,

		defineParams: function () {
			// free function, called once ever; returns parameter definition (hierarchical) array
			return [
				{
					id: "draggableHandles", type: "handle", uiName: "$$$/animal/Behavior/RigDynamics/Parameter/draggableHandles=Draggable Handles",
					dephault: { match: "//Adobe.Dragger.Draggable" }, hidden:true
				},
				{
					id: "fixedHandles", type: "handle", uiName: "$$$/animal/Behavior/RigDynamics/Parameter/fixedHandles=Fixed Handles",
					dephault: { match: "//Adobe.HandleFixer.Fixed" }, hidden:true
				},
				{
					id: "gravityStrength", type: "slider", uiName: "$$$/animal/Behavior/RigDynamics/Parameter/gravityStrength=Gravity Strength", uiUnits: "%", min: 0, precision: 1, dephault: 10
				},
				{
					id: "gravityDirection", type: "angle", uiName: "$$$/animal/Behavior/RigDynamics/Parameter/gravityDirection=Gravity Direction", precision: 0, dephault: 180
				},
				{
					id: "dangleGroup", type: "group",
					uiName: "$$$/animal/Behavior/RigDynamics/Parameter/dangleGroup=Dangle",
                    uiToolTip: "$$$/animal/Behavior/RigDynamics/Parameter/dangleGroup/tooltip=Settings specific to Dangle-tagged handles",
					groupChildren: defineDangleParams()
				},
				{
					id: "collisionGroup", type: "group",
					uiName: "$$$/animal/Behavior/RigDynamics/Parameter/collisionGroup=Collision",
                    uiToolTip: "$$$/animal/Behavior/RigDynamics/Parameter/collisionGroup/tooltip=Settings specific to Collide- and Dynamic-tagged layers",
					groupChildren: defineCollisionParams()
				}
			];
		},

		defineTags: function () {
			 return {
				aTags: [
					{
						id: "Adobe.RigDynamics.Dangle",
						artMatches: ["dangle"],
						uiName: "$$$/animal/Behavior/RigDynamics/UIName/Dangle=Dangle",
						tagType: "handletag",
						uiGroups: [{ id:"Adobe.TagGroup.Physics"}]
					},
					{
						id: "Adobe.Dragger.Draggable",
						artMatches: ["draggable", "mousetrack"],
						uiName: "$$$/animal/Behavior/Draggable/UIName/Draggable=Draggable",
						tagType: "handletag",
						uiGroups: [{ id:"Adobe.TagGroup.Modifiers"}]
					},
					{
						id: "Adobe.HandleFixer.Fixed",
						artMatches: ["fixed"],
						uiName: "$$$/animal/Behavior/HandleFixer/UIName/Fixed=Fixed",
						tagType: "handletag",
						uiGroups: [{ id:"Adobe.TagGroup.Modifiers"}]
					},
					{
						id: "Adobe.Physics.Collide",
						artMatches: ["collide"],
						uiName: "$$$/animal/Behavior/Box2d/World/TagName/Collide=Collide",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.Physics"}]
					},
					{
						id: "Adobe.Physics.Dynamic",
						artMatches: ["dynamic"],
						uiName: "$$$/animal/Behavior/Box2d/World/TagName/Dynamic=Dynamic",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.Physics"}],
						// note copy of "$$$/animal/Tag/ToolTip/UsedBy=Used by behavior ^[@0^]" here, because it skipped when you specify your own tooltip
						uiToolTip: "$$$/animal/Behavior/Box2d/World/Tag/Dynamic/Tooltip=Used by behavior ^[Physics^]\nUsually combined with Collide tag; incompatible with Dangle tag"
					},
				]
			};
		},

		onCreateBackStageBehavior: function (/*self*/) {
			 return {
				 order : 1.1, // Needs to run after dragger and handle fixer
			 };
		},

		onCreateStageBehavior: function (self, args) {
			onCreateRigDynamicsStageBehavior(self, args);
			onCreateCollisionStageBehavior(self, args);
		},

		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage

			onRigDynamicsAnimate(self, args);
			onCollisionAnimate(self, args);

		}
	}; // end of object being returned
});
